<?php
/**
 * Plugin Name: Easy Age Gate Plus
 * Description: Restrict access to users over 18 with fully customizable popup.
 * Version: 1.2
 * Author: Anna Nicole Kucerova
 * Author URI: https://nicoledigital.cz
 */

if (!defined('ABSPATH')) exit;

// Admin menu
function easy_age_gate_plus_menu() {
    add_menu_page(
        'EasyAgeGatePlus Settings',
        'EasyAgeGatePlus',
        'manage_options',
        'easy-age-gate-plus',
        'easy_age_gate_plus_settings_page',
        'dashicons-shield',
        20
    );
}
add_action('admin_menu', 'easy_age_gate_plus_menu');

// Settings page
function easy_age_gate_plus_settings_page() {
    ?>
    <div class="wrap">
        <h1>EasyAgeGatePlus CZ</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('easy_age_gate_plus_settings');
            do_settings_sections('easy-age-gate-plus');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

// Register settings
function easy_age_gate_plus_register_settings() {
    $options = [
        'enter_button_text',
        'exit_button_text',
        'exit_redirect_url',
        'popup_heading',
        'conditions_text',
        'terms_text'
    ];

    foreach ($options as $opt) {
        register_setting('easy_age_gate_plus_settings', "easy_age_gate_plus_$opt");
    }

    add_settings_section('main_section', 'Popup Settings', null, 'easy-age-gate-plus');

    add_settings_field('enter_button_text', 'Enter button text', 'easy_age_gate_plus_text_cb', 'easy-age-gate-plus', 'main_section', ['id' => 'enter_button_text', 'default' => 'Enter']);
    add_settings_field('exit_button_text', 'Exit button text', 'easy_age_gate_plus_text_cb', 'easy-age-gate-plus', 'main_section', ['id' => 'exit_button_text', 'default' => 'Leave']);
    add_settings_field('exit_redirect_url', 'Exit redirect URL', 'easy_age_gate_plus_text_cb', 'easy-age-gate-plus', 'main_section', ['id' => 'exit_redirect_url', 'default' => 'https://google.com']);
    add_settings_field('popup_heading', 'Popup heading', 'easy_age_gate_plus_text_cb', 'easy-age-gate-plus', 'main_section', ['id' => 'popup_heading', 'default' => 'Age Verification']);
    add_settings_field('conditions_text', 'Conditions text', 'easy_age_gate_plus_textarea_cb', 'easy-age-gate-plus', 'main_section', ['id' => 'conditions_text', 'default' => 'You must be over 18 to enter.']);
    add_settings_field('terms_text', 'Terms of Use text', 'easy_age_gate_plus_textarea_cb', 'easy-age-gate-plus', 'main_section', ['id' => 'terms_text', 'default' => 'By entering, you agree to our Terms of Use.']);
}
add_action('admin_init', 'easy_age_gate_plus_register_settings');

function easy_age_gate_plus_text_cb($args) {
    $value = get_option("easy_age_gate_plus_{$args['id']}", $args['default']);
    echo '<input type="text" style="width: 100%;" name="easy_age_gate_plus_' . esc_attr($args['id']) . '" value="' . esc_attr($value) . '">';
}

function easy_age_gate_plus_textarea_cb($args) {
    $value = get_option("easy_age_gate_plus_{$args['id']}", $args['default']);
    echo '<textarea style="width: 100%;" rows="4" name="easy_age_gate_plus_' . esc_attr($args['id']) . '">' . esc_textarea($value) . '</textarea>';
}

// Frontend popup
function easy_age_gate_plus_popup_markup() {
    if (!isset($_COOKIE['easy_age_gate_plus_accepted'])) {
        $heading   = get_option('easy_age_gate_plus_popup_heading', 'Age Verification');
        $text      = get_option('easy_age_gate_plus_conditions_text', 'You must be over 18 to enter.');
        $terms     = get_option('easy_age_gate_plus_terms_text', '');
        $enter     = get_option('easy_age_gate_plus_enter_button_text', 'Enter');
        $exit      = get_option('easy_age_gate_plus_exit_button_text', 'Leave');
        $exit_url  = get_option('easy_age_gate_plus_exit_redirect_url', 'https://google.com');

        echo "
        <div id='easy-age-gate-popup'>
            <div class='popup-content'>
                <h2>$heading</h2>
                <p>$text</p>
                " . ($terms ? "<small>$terms</small>" : "") . "
                <div class='popup-buttons'>
                    <button id='age-gate-enter'>$enter</button>
                    <button id='age-gate-exit' data-url='$exit_url'>$exit</button>
                </div>
            </div>
        </div>";
    }
}
add_action('wp_footer', 'easy_age_gate_plus_popup_markup');

// Styles + JS
function easy_age_gate_plus_assets() {
    ?>
    <style>
        #easy-age-gate-popup {
            position: fixed;
            top: 0; left: 0;
            width: 100%; height: 100%;
            background: white;
            color: white;
            z-index: 9999;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .popup-content {
            text-align: center;
            padding: 30px;
            border: 1px solid white;
            background: #ff5733;
            border-radius: 10px;
        }
        .popup-content h2 { margin-bottom: 10px; }
        .popup-content p, .popup-content small { margin: 10px 0; }
        .popup-buttons button {
            padding: 10px 20px;
            margin: 10px;
            border: none;
            font-size: 16px;
            cursor: pointer;
            color: white;
        }
        #age-gate-enter { background-color: green; }
        #age-gate-exit { background-color: red; }
    </style>
    <script>
        document.addEventListener('DOMContentLoaded', function () {
            const enterBtn = document.getElementById('age-gate-enter');
            const exitBtn = document.getElementById('age-gate-exit');

            if (enterBtn) {
                enterBtn.addEventListener('click', () => {
                    document.getElementById('easy-age-gate-popup').style.display = 'none';
                    document.cookie = "easy_age_gate_plus_accepted=true; path=/; max-age=31536000";
                });
            }

            if (exitBtn) {
                exitBtn.addEventListener('click', () => {
                    const url = exitBtn.getAttribute('data-url') || 'https://google.com';
                    window.location.href = url;
                });
            }
        });
    </script>
    <?php
}
add_action('wp_head', 'easy_age_gate_plus_assets');

